#!/usr/bin/env python
# Copyright (c) TripleBlind Holdings, Inc. Confidential and Proprietary. All rights reserved.
import os

import torch

import tripleblind as tb


##########################################################################
# GET AUTHENTICATION TOKENS AND ESTABLISH CONNECTION TO THE ROUTER
#
# IN THIS INSTANCE WE ARE TREATING ORGANIZATION-ONE AS "SANTANDER"
#
# Establish the connection details to reach the TripleBlind instance.
# Unless explicitly specified, all operations will occur via this default
# session as the user 'organization_one'
tb.initialize(api_token=tb.config.example_user3["token"], example=True)

#############################################################################
# Validate that the datasets are available in the Router index

prefix = "TEST" if "TB_TEST_SMALL" in os.environ else "EXAMPLE"

dataset_train0 = tb.TableAsset.find(
    f"{prefix} - SAN Customer Database", owned_by=tb.config.example_user1["team_id"]
)
dataset_train1 = tb.TableAsset.find(
    f"{prefix} - JPM Customer Database", owned_by=tb.config.example_user2["team_id"]
)
dataset_train2 = tb.TableAsset.find(
    f"{prefix} - PNB Customer Database", owned_by=tb.config.example_user3["team_id"]
)
if not dataset_train0 or not dataset_train1 or not dataset_train2:
    raise SystemExit("Datasets not found.")

# Find the existing neural network in the Router index
asset_id = tb.util.load_from("model_asset_id.out")
if not asset_id:
    raise SystemExit("You must first run 1_position_on_accesspoint.py.")
alg_santandar = tb.Asset(asset_id)

#############################################################################
# Designate the files to use and train the network

# Loss function names must be consistent with PyTorch.
#   See: https://pytorch.org/docs/stable/nn.html#loss-functions
# Currently tested: 'BCEWithLogitsLoss', 'NLLLoss', 'CrossEntropyLoss'
loss_name = "BCEWithLogitsLoss"
pos_weight = tb.TorchEncoder.encode(torch.arange(17, 18, dtype=torch.int32))

# Optimizer names must be consistent with PyTorch.
#   See: https://pytorch.org/docs/stable/optim.html
# Currently tested: 'SGD', 'Adam', 'Adadelta'
optimizer_name = "Adam"
optimizer_params = {"lr": 0.001}

# Use the CSV Preprocessor to specify what data to use for training and which
# column to treat as the classification label.
csv_pre = (
    tb.TabularPreprocessor.builder()
    .add_column("target", target=True)
    .all_columns(True)
    .dtype("float32")
)

job = tb.create_job(
    job_name=f"Transfer_Learning - {tb.util.timestamp()}",
    operation=alg_santandar,
    dataset=[dataset_train0, dataset_train1, dataset_train2],
    preprocessor=csv_pre,
    params={
        "epochs": 1,
        "loss_meta": {"name": loss_name, "params": {"pos_weight": pos_weight}},
        "optimizer_meta": {"name": optimizer_name, "params": optimizer_params},
        "data_type": "table",
        "data_shape": [26],  # number of columns of data in table
        "model_output": "binary",  # binary/multiclass/regression
        "split_transfer_train": True,
        "split_layer": 3,
    },
)

print("Training network...")

#############################################################################
# Create the network asset and local .pth file from the trained network
#
if job.submit():
    job.wait_for_completion()
    # Throw away this network definition (no longer needed)
    if job.success:
        print()
        print("Trained Network Asset ID:")
        print("    ===============================================")
        print(f"    ===>  {job.result.asset.uuid} <===")
        print("    ===============================================")
        print("    Algorithm: Deep Learning Model")
        print(f"    Job ID:    {job.job_name}")
        print()
        trained_network = job.result.asset
    else:
        raise SystemExit("Training failed")

    # Pull down the model for local validation
    local_filename = trained_network.retrieve(save_as="local.zip", overwrite=True)
    print("Trained network has been downloaded as:")
    print(f"   {local_filename}")

    # Save filename for use in 3a_local_inference.py
    tb.util.save_to("local_model_filename.out", local_filename)
    # Save asset ID for use in 3b_aes_inference.py / 3c_smpc_inference.py
    tb.util.save_to("model_asset_id.out", job.result.asset.uuid)

    print("Ready to run local inference.")
    print()

    # Create an agreement which allows the other teams to use this
    # trained model in subsequent steps.
    agreement = job.result.asset.add_agreement(
        with_team=tb.config.example_user2["team_id"], operation=tb.Operation.EXECUTE
    )
    if agreement:
        print("Created Agreement for use of trained Asset.")

############################################################################
# The 'local.pth' file now holds the refined version of the PyTorch model.
# It can now easily be used for local inference or submitted for further
# training and refinement.
