#!/usr/bin/env Rscript
# Copyright (c) TripleBlind Holdings, Inc. Confidential and Proprietary. All rights reserved.

# Install required packages if not already installed
RequiredPackages <- c("reticulate", "stringr")
for (i in RequiredPackages) { # Installs packages if not yet installed
  if (!require(i, character.only = TRUE)) install.packages(i)
}

# Load packages
library(reticulate)
library(stringr)

pd <- import("pandas", convert = FALSE)

sklearn_metrics <- import("sklearn.metrics")
classification_report <- sklearn_metrics$classification_report

tb <- import("tripleblind")

tb$util$set_script_dir_current()
data_dir <- file.path("example_data")

# This step downloads files necessary to run the example
tb$util$download_tripleblind_resource(
  "Sant_test_small.pkl",
  save_to_dir=data_dir,
  cache_dir="../../../.cache"
)
tb$util$download_tripleblind_resource(
  "Sant_test_small.csv",
  save_to_dir=data_dir,
  cache_dir="../../../.cache"
)

##########################################################################
# GET AUTHENTICATION TOKENS AND ESTABLISH CONNECTION TO THE ROUTER
#
tb$initialize(api_token = tb$config$example_user2[["token"]])

# with(open("model_asset_id.out", open = "r"))

result <- tryCatch(
  {
    f <- file("model_asset_id.out", open = "r")
    asset_id <- trimws(readLines(f, warn = FALSE)) # trimws is R equivalent of python strip(); removes leading and trailing whitespace
    close(f)
  },
  error = function(err) {
    print("You must run 2_model_train.py first.")
    quit(save = "ask")
  }
)

alg_santandar <- tb$Asset(asset_id)

testdf <- pd$read_csv(file.path(data_dir, "Sant_test_small.csv"))
testdf <- testdf$drop(columns = "target")

testdf$to_csv(file.path(data_dir, "Sant_test_small_notarget.csv"), index = FALSE)
datasets <- list(file.path(data_dir, "Sant_test_small_notarget.csv"))

# Define a job using this model
job <- tb$create_job(
  job_name = paste("Santandar Inference - ", toString(Sys.time()), sep = ""),
  operation = alg_santandar,
  dataset = datasets,
  params = list("security" = "smpc") # fed or smpc
)

if (is.null(job)) {
  print("ERROR: Failed to create the job -- do you have an Agreement to run this?")
  cat("\n")
  print(
    paste("NOTE: Remote inference requires the user '", tb$config$example_user1[["login"]], "' create an", sep = "")
  )
  print(
    paste("      Agreement on their algorithm asset with user '", tb$config$example_user2[["login"]], "'", sep = "")
  )
  print(
    paste("      (", tb$config$example_user2[["name"]], ") before they can use it to infer.  You can do", sep = "")
  )
  print("      this on the Router at:")
  print(paste("      ", tb$config$gui_url, "/dashboard/algorithm/", alg_santandar$uuid, sep = ""))
  cat("\n")
  print("Program exiting...")
  quit(save = "ask")
}

p <- pd$read_pickle(file.path(data_dir, "Sant_test_small.pkl"))
y <- p[["y"]]

if (job$submit()) {
  job$wait_for_completion()

  Sys.sleep(3)
  if (job$success) {
    filename <- job$result$asset$download(
      save_as = "smpc_out.zip", overwrite = TRUE
    )
    pack <- tb$Package$load(filename)
    result <- pd$read_csv(pack$record_data_as_file(), names = list("results"))
    cat("\n")
    print("Inference results:")
    print(paste(py_to_r(result$values), collapse = "  "))
    cat("\n")

    print("Truth:")
    print(paste(py_to_r(y), collapse = "  "))
    cat("\n")

    print("Classification report:")
    report <- classification_report(y, result$values)
    cat(report)
    cat("\n")
  } else {
    print("Inference failed")
  }
}
