#!/usr/bin/env python
# Copyright (c) TripleBlind Holdings, Inc. Confidential and Proprietary. All rights reserved.
from pathlib import Path

import tripleblind as tb


name0 = "EXAMPLE - Split Statistics (part 1)"
name1 = "EXAMPLE - Split Statistics (part 2)"
name2 = "EXAMPLE - Split Statistics (part 3)"
desc = """Synthetic data for Split Record Statistics example

This group of three discrete datasets look something like this:

EXAMPLE - Split Statistics (part 1)
  identifier, Normal,     Continuous
  1,         -0.11,       0.1111
  2,          0.22,       0.2222
  3,          1.33,       0.3333

EXAMPLE - Split Statistics (part 2)
  id,         Continuous, DiscreteA
  1,          0.1212121,  11
  3,          0.3232323,  33

EXAMPLE - Split Statistics (part 3)
  id,         DiscreteA,  DiscreteB
  1,          311,        3113
  3,          333,        3333

These can be combined and thought of as single virtual table of records that
look like this:

  id, Normal, 0.Continuous, 1.Continuous, 1.DiscreteA, 2.DiscreteA, DiscreteB
  1, -0.11,   0.1111,       0.1212121,    11,          311,         3113
  3,  1.33,   0.3333,       0.3232323,    33,          333,         3333

See example/2_get_split_record_statistics.py in the SDK for details on how
to calculate statistics on this data.
"""


tb.util.set_script_dir_current()
data_dir = Path("example_data")
tb.initialize(api_token=tb.config.example_user1["token"])

# Register the first database with the Router and place on Access Point
try:
    # Start a connection as 'organization_one' user
    session1 = tb.Session(api_token=tb.config.example_user1["token"], from_default=True)

    print(f"Creating '{name0}' on {tb.config.example_user1['login']}'s Access Point...")
    dataset0 = tb.Asset.position(
        file_handle=data_dir / "dataset0.csv",
        name=name0,
        desc=desc,
        is_discoverable=True,
        k_grouping=5,
        session=session1,
    )
    if tb.config.create_agreements_on_example_input_assets:
        # Give user3 access to this file for the STATS calculation
        dataset0.add_agreement(
            with_team=tb.config.example_user3["team_id"],
            operation=tb.Operation.STATS,
            session=session1,
        )
        print("Created Agreement")

except tb.TripleblindAssetAlreadyExists:
    print(f"Asset '{name0}' already exists, skipping.")


# Register the second dataset with the Router and place on Access Point
try:
    # Start a connection as 'organization_two' user
    session2 = tb.Session(api_token=tb.config.example_user2["token"], from_default=True)

    print(f"Creating '{name1}' on {tb.config.example_user2['login']}'s Access Point...")
    dataset1 = tb.Asset.position(
        file_handle=data_dir / "dataset1.csv",
        name=name1,
        desc=desc,
        is_discoverable=True,
        session=session2,
    )
    if tb.config.create_agreements_on_example_input_assets:
        # Give user3 access to this file for the STATS calculation
        dataset1.add_agreement(
            with_team=tb.config.example_user3["team_id"],
            operation=tb.Operation.STATS,
            session=session2,
        )
        print("Created Agreement")
except tb.TripleblindAssetAlreadyExists:
    print(f"Asset '{name1}' already exists, skipping.")


# Register the third dataset with the Router and place on Access Point
try:
    # Start a connection as 'organization_three' user
    session3 = tb.Session(api_token=tb.config.example_user3["token"], from_default=True)

    print(f"Creating '{name2}' on {tb.config.example_user3['login']}'s Access Point...")
    dataset2 = tb.Asset.position(
        file_handle=data_dir / "dataset2.csv",
        name=name2,
        desc=desc,
        is_discoverable=True,
        k_grouping=5,
        session=session3,
    )

except tb.TripleblindAssetAlreadyExists:
    print(f"Asset '{name2}' already exists, skipping.")

print("Data is in position.")
