[CmdletBinding()]
param (
    [string]$token,
    [string]$cert,
    [string]$yaml,
    [switch]$nossl = $false,
    [switch]$ni = $false,
    [switch]$help = $false
    # [switch]$debug = $false # built into PS as $DebugPreference
)


# makes CommandNotFound and others terminating to try-catchs
$oldPreference = $ErrorActionPreference
$ErrorActionPreference = 'stop'


# store coloured unicode symbols, check and X
function Ok { Write-Host "  $([char]0x2713) " -ForegroundColor Green -NoNewline }
function Err { Write-Host "  $([char]0x2718) " -ForegroundColor Red -NoNewline }
# an exclamation mark is used for action items
function Excl { Write-Host "  ! " -ForegroundColor Red -NoNewline }
# gracefully exit the script with cleanup on failure
function ExitSetup {
    $ErrorActionPreference = $oldPreference
    if ($DebugPreference) {
        Write-Host '    Debugging log can be found in the SDK folder as debug.txt'
        Stop-Transcript
    }
    Err; Write-Host -NoNewLine 'Installation not successful, press any key to exit...';
    $null = $Host.UI.RawUI.ReadKey('NoEcho,IncludeKeyDown');
    exit
}

# check for PowerShell version; we only support Powershell 5.1 and newer
try {
    $psversion = $PSVersionTable.PSVersion
    if (-Not ($psversion.Major -ge 5)) {
        throw "PowerShell version too old"
    }
} catch {
    Write-Host ' '
    Err; Write-Host 'PowerShell Version >= 5 required. Please update your PowerShell.'
    Write-Host '    Alternatively, you can use the backup `install.bat` installer.'
    ExitSetup
}


# force current directory to be script location
Set-Location $PSScriptRoot
# CONSIDER: https://github.com/PowerShell/PowerShell/issues/10994
# the transcript will contain minimal actually useful information; remove?
if ($DebugPreference) { Start-Transcript debug.txt }
#### HELPERS ####
# $sdkver = 1.43.0 # not needed
$installConda = $false
$everythingOk = $true


#### HELP ###
if ($help) {
    Write-Host ' '
    Write-Host '    Welcome to the TripleBlind installer Help section'
    Write-Host '    This script will install TripleBlind on your system,'
    Write-Host '    diagnosing potential issues encountered.'
    Write-Host '    TripleBlind will be installed as a Conda environment.'
    Write-Host '    Miniconda will be installed if no Anaconda installation is found.'
    Write-Host ' '
    Write-Host '   .\install.ps1 [-help -debug -ni -nossl -token <string> -cert <path> -yaml <path>]'
    Write-Host ' '
    Write-Host '    -ni             Non-interactive mode flag. Can still pass -token.'
    Write-Host '    -nossl          Flag to bypass SSL validation during install.'
    Write-Host '    -debug          Flag to print extra info and output to debug.txt file.'
    Write-Host '    -token [string] Optionally pass user SDK token directly.'
    Write-Host '    -cert [path]    If a company root certificate should be used, path to'
    Write-Host '                    the .pem file. If it is .crt, you need to covert it first.'
    Write-Host '    -yaml [path]    Path to alternate custom yaml file to build tripleblind env,'
    Write-Host '                    this should be based on the environment-win.yaml file.'
    Write-Host ' '

    # offer to continue installation or quit the script
    $title    = ''
    $question = '    Do you want to being the installation?'
    $choices  = '&Yes, start TripleBlind installation.', '&No, close the installation script.'
    $decision = $Host.UI.PromptForChoice($title, $question, $choices, 1)
    if ($decision -eq 1) {
        exit
    }
}

# validate $yaml and $cert file inputs
if ($yaml) {
    $yamlExt = (Get-ChildItem $yaml).Extension
    if ($yamlExt -ne ".yaml") {
        Err; Write-Host "The conda environment file must be .yaml, received $yamlExt"
        ExitSetup
    }
}
if ($cert) {
    $certExt = (Get-ChildItem $cert).Extension
    if ($certExt -ne ".pem") {
        Err; Write-Host "The SSL certificate file must be .pem, received $certExt"
        ExitSetup
    }
}

#### INTRO ####
# display intro graphics
Write-Host ' '
Write-Host '                               ***' -ForegroundColor DarkCyan
Write-Host '                              *****' -ForegroundColor DarkCyan
Write-Host '                               ***' -ForegroundColor DarkCyan
Write-Host ' '
Write-Host '                          *** ' -ForegroundColor DarkBlue -NoNewline
Write-Host '      ***' -ForegroundColor Blue
Write-Host '                         *****' -ForegroundColor DarkBlue -NoNewline
Write-Host '     *****' -ForegroundColor Blue
Write-Host '                          *** ' -ForegroundColor DarkBlue -NoNewline
Write-Host '      ***' -ForegroundColor Blue

Write-Host ' '
Write-Host '    This script will prepare your system to use the TripleBlind SDK'
Excl; Write-Host 'Before starting, please make sure to close any programs or IDEs'
Write-Host '    that might be using conda or the tripleblind environment.'
Write-Host '    ----------------'
# ensure the message gets read
Start-Sleep 10


# we confirmed powershell version above, display it here for consistency
Write-Host ' '
Ok; Write-Host "PowerShell Version compatible: $psversion"

#### PROCESSOR ####
# check the processor architecture and give a warning if on `ARM64`
$arch = $env:PROCESSOR_ARCHITECTURE
$compatibleArch = 'AMD64', 'IA64', 'EM64T', 'X86'
if ($compatibleArch -contains $arch) {
    Ok; Write-Host "Processor architecture compatible: $arch"
} else {
    Excl; Write-Host "Processor architecture might not be compatible: $arch"
    $everythingOk = $false
}


#### CONDA ####
# check if conda is available from path
try {
    # saved to variable to avoid unnecessary print
    $condaComm = Get-Command conda
    & conda activate base
    $condaVer = & conda --version
    $condaVer = $condaVer.Substring(6)
    Ok; Write-Host "conda command version $condaVer available"

} catch [System.Management.Automation.CommandNotFoundException] {
    # Initialize variables for Conda installation paths
    $condaLocations = @()

    # Function to check if Conda is installed in a given directory
    function Test-CondaInstallation($directory) {
        $condaExePath = Join-Path $directory "Scripts\conda.exe"
        if (Test-Path -Path $condaExePath) {
            return $directory
        }
        return $null
    }

    # Define common installation paths for Conda
    $minicondaPath = Join-Path $env:USERPROFILE "miniconda3"
    $minicondaPath = Test-CondaInstallation $minicondaPath
    $anacondaPath = Join-Path $env:USERPROFILE "anaconda3"
    $anacondaPath = Test-CondaInstallation $anacondaPath
    $programDataMinicondaPath = Test-CondaInstallation "C:\ProgramData\miniconda3"
    $programDataAnacondaPath = Test-CondaInstallation "C:\ProgramData\anaconda3"

    # Check each installation path for Conda and add to $condaLocations if installed
    if ($minicondaPath -ne $null) {
        $condaLocations += Test-CondaInstallation $minicondaPath
    }
    if ($anacondaPath -ne $null) {
        $condaLocations += Test-CondaInstallation $anacondaPath
    }
    if ($programDataMinicondaPath -ne $null) {
        $condaLocations += Test-CondaInstallation $programDataMinicondaPath
    }
    if ($programDataAnacondaPath -ne $null) {
        $condaLocations += Test-CondaInstallation $programDataAnacondaPath
    }

    # Conda installation found
    if ($condaLocations.Length -gt 0) {
        # if there's more than one potential installation location, prompt the user to choose the right one
        if ($condaLocations.Length -gt 1) {

            Excl; Write-Host 'Multiple conda locations detected'
            $question = '    Which of the following locations is your conda installation?'
            $choices  = $condaLocations | ForEach-Object { $_ }
            # give choices hot-key numbers for the user selection
            for ($i = 0; $i -lt $choices.Length; $i++)
            {
                $prepend = '&' + ($i+1).ToString() + ' '
                $choices[$i] = $choices[$i] -replace '^', $prepend
            }

            $decision = $Host.UI.PromptForChoice('', $question, $choices, 0)
            $condaLocation = $condaLocations[$decision]

        } else {
            # if there's just one registry entry, choose it as location
            $condaLocation = $condaLocations[0]
        }

        Ok; Write-Host "Anaconda distribution installation found at $condaLocation"
        Write-Host '    conda command not available from path'
        Write-Host "    Activating conda using $condaLocation\shell\condabin\conda-hook.ps1"

        # append the specific powershell activation file path
        # C:\Users\XXX\Anaconda3\shell\condabin\conda-hook.ps1
        $condaPSHook = "$condaLocation\shell\condabin\conda-hook.ps1"

        # check that the path exists
        if (Test-Path $condaPSHook) {
            # call the activation script
            & $condaPSHook
            & conda activate $condaLocation
            & conda init --all --quiet

            Ok; Write-Host 'conda command now available'
            Write-Host '    conda has been initiated for all available shells'

        # otherwise this is probably just a post-uninstall stub in registry
        # we try to install a new conda over it
        } else {
            Excl; Write-Host 'conda activation was not successful'

            # ask the user to decide how to proceed
            # the options are either to close the terminal and rerun from anaconda prompt if they hsve it
            # or continue with our installer setting up conda for them
            $question = '    Do you have Anaconda installed on your machine?'
            $choices  = '&Yes', '&No'
            $decision = $Host.UI.PromptForChoice('', $question, $choices, 1)
            if ($decision -eq 0) {
                Write-Host '    Unfortunately, the installation script was not able to detect it.'
                Excl; Write-Host 'Please open Anaconda Powershell Prompt and navigate to the SDK folder.'
                Write-Host '    Then run .\install.ps1'
                ExitSetup

            } else {
                Ok; Write-Host 'conda installation was not found'
                Write-Host '    The installer will install miniconda for you.'
                $installConda = $true
            }
        }
    } else {
        Ok; Write-Host 'conda installation not found'
        Write-Host '    The installer will install miniconda for you.'
        $installConda = $true
    }
}


# install conda if missing
if ($installConda) {
    Write-Host ' '
    Excl; Write-Host 'Anaconda was not found.  It is highly recommended that you use a virtual'
    Write-Host '    environment, and Anaconda provides that plus a package management system.'
    Write-Host '    This script can download the latest version of Miniconda and install it.'
    Excl; Write-Host 'If you have already installed Anaconda, you should cancel, open the'
    Write-Host '    `Anaconda Powershell Prompt` from your Windows menu, then resume the setup.'
    Write-Host ' '
    Write-Host '    If you do not wish to use Anaconda, you can cancel this installation.  You will'
    Write-Host '    need to manually install the tripleblind.whl files and dependencies.'
    Write-Host ' '

    if (-not $ni) {
        # give the option to close the script without installation
        $question = '    Can we proceed to install Miniconda?'
        $choices  = '&Yes, install Miniconda for me', '&No, close the setup'

        $decision = $Host.UI.PromptForChoice('', $question, $choices, 0)
        if ($decision -eq 1) {
            ExitSetup
        }
    }

    Write-Host ' '
    Write-Host '    Downloading Miniconda...'
    if (-not (Get-Module -ListAvailable -Name BitsTransfer)) {
        Import-Module BitsTransfer
    }
    $condaSrc = 'https://repo.anaconda.com/miniconda/Miniconda3-py39_4.12.0-Windows-x86_64.exe'

    # catch a download error
    try {
        Start-BitsTransfer -Source $condaSrc -Destination $PSScriptRoot
    } catch {
        Err; Write-Host "Error while downloading Miniconda installer: $_"
        ExitSetup
    }

    # catch an installation error
    $condaLocation = "$env:USERPROFILE\Miniconda3"
    Write-Host "    conda will be installed into $condaLocation"
    try {
        Start-Process $PSScriptRoot'\Miniconda3-py39_4.12.0-Windows-x86_64.exe' -Wait `
                -ArgumentList @('/S', '/InstallationType=JustMe', 'RegisterPython=0', "/D=$condaLocation")
    } catch {
        Err; Write-Host "Error while installing Miniconda: $_"
        ExitSetup
    }

    Ok; Write-Host 'conda installed successfully'
    Write-Host '    conda will be activated and initiated'
    # initiate and activate conda
    & $condaLocation\shell\condabin\conda-hook.ps1
    & conda activate $condaLocation
    & conda init --all --quiet

    Ok; Write-Host 'conda command now available'
    Write-Host '    conda has been initiated for all available shells'
}
# TODO: understand if this could pose an issue on locked down systems
# https://github.com/conda/conda/issues/8428#issuecomment-518089508


#### TRIPLEBLIND ####
# at this point, conda is either available or the script was terminated
# if tripleblind environment exists, offer to upgrade
$createEnv = $false
& conda activate base
# equivalent to conda env list but gives a nicer PS object
$condaEnvList = Get-CondaEnvironment
if ($condaEnvList.Name -contains 'tripleblind') {
    Ok; Write-Host 'An existing "tripleblind" environment was found.'

    # offer upgrade unless the -ni flag is passed
    if (-not $ni) {
        Write-Host '    Starting from a clean environment is generally safest and fastest.'
        Write-Host '    However, this will remove any extra packages you might have'
        Write-Host '    installed manually using pip or conda commands.'
        Write-Host '    You will need to reinstall any such packages after this update.'

        $question = '    Do you want to start from a clean environment?'
        $choices  = '&Create a clean tripleblind env', '&Upgrade the existing env'
        $decision = $Host.UI.PromptForChoice('', $question, $choices, 1)

        if ($decision -eq 0) {
            # clean the environment
            & conda env remove --name tripleblind
            $createEnv = $true
        }
    }

} else {
    $createEnv = $true
}

# prepare conda --flags based on user preference
$condaFlag = @()
if ($nossl) {$condaFlag += '--insecure'}
if ($DebugPreference) {
    $condaFlag += '--verbose'
    $pipFlag = @()
} else { $pipFlag = '--quiet' }

if ($createEnv) {
    Write-Host "    Creating a new 'tripleblind' environment"
    try {
        & conda create $condaFlag --name tripleblind python=3.10 -y
        Ok; Write-Host "'tripleblind' environment created"
    } catch {
        Err; Write-Host "Error while creating 'tripleblind' env: $_"
        ExitSetup
    }
}


try {
    & conda activate tripleblind
    # PS equivalent is Enter-CondaEnvironment tripleblind
} catch {
    Err; Write-Host "Error while activating 'tripleblind' env: $_"
    ExitSetup
}


# use default environment-win.yaml file unless passed a custom one
if (-not $yaml) {$yaml = 'environment-win.yaml'}

# upgrade the environment from empty env or from existing env (same procedure)
try {
    if ($cert) {
        # if a company certificate path is passed, use it during the install
        Write-Host "    Setting conda and pip SSL certificates to $cert"
        & conda config --set ssl_verify $cert
        & pip config set global.cert $cert
        & conda env update $condaFlag --name tripleblind -f $yaml
        # & conda config --remove-key ssl_verify
        # & pip config unset global.cert
    } elseif ($nossl) {
        & conda --env config vars set PYTHONUTF8=1 > nul
        & conda --env config vars set PYTHONIOENCODING=utf-8 > nul

        # parse the $yaml file for conda dependencies and then for pip dependencies
        $dep = Get-Content $yaml
        $condaDep = $dep[($dep.IndexOf('dependencies:')+2)..($dep.IndexOf('  - pip:')-1)].Substring(4)
        $pipDep = $dep[($dep.IndexOf('  - pip:')+1)..($dep.IndexOf('variables:')-1)].Substring(8)

        # "$var" turns array $var into space-separated string
        & conda install -y $condaFlag "$condaDep"
        & pip install $pipFlag --trusted-host pypi.org --trusted-host pypi.python.org --trusted-host files.pythonhosted.org "$pipDep"
    } else {
        & conda env update $condaFlag --name tripleblind -f $yaml
    }
} catch {
    Err; Write-Host "Error while installing dependencies: $_"
    Write-Host '    TripleBlind may not work properly'
    $everythingOk = $false
}

# check that all packages installed
Write-Host '    Verifying that all packages installed properly.'

$condaList = & conda list
# make sure seeing `\envs\tripleblind` in the path doesn't trip the install check
$condaList = $condaList[3..($condaList.Count-1)]
# check for our two packages, tripleblind & preprocessor
if (-not [bool]($condaList -match 'tripleblind')) {
    Err; Write-Host 'Unfortunately the tripleblind package failed to install.'
    Write-Host '    Please review the output to determine the source of the error.'
    ExitSetup
} else { Ok; Write-Host 'tripleblind package installed successfully' }
if (-not [bool]($condaList -match 'preprocessor')) {
    Err; Write-Host 'Unfortunately the preprocessor package failed to install.'
    Write-Host '    Please review the output to determine the source of the error.'
    ExitSetup
} else { Ok; Write-Host 'preprocessor package installed successfully' }

# for other packages match the package name precisely from each line in yaml file
$rx = '(?<=::|- )(?!defaults)\b[A-z][-A-z0-9]+(?==|<|>|$)'
$depList = (Select-String $rx -AllMatches $yaml).Matches.Value

foreach ($pkg in $depList) {
    if (-not [bool]($condaList -match $pkg)) {
        Err; Write-Host "$pkg not installed properly"
        $everythingOk = $false
    } elseif ($DebugPreference) {
        Ok; Write-Host "$pkg installed properly"
    }
}

#### YAML FILE SETUP ####
if (-not $token -and -not $ni) {
    Write-Host ' '
    Excl; Write-Host 'In order to associate the SDK with your user account,'
    Write-Host '    you must use your personalized `tripleblind.yaml` file.'
    Write-Host '    To collect your `tripleblind.yaml` file:'
    Write-Host '     - Browse to https://tripleblind.app and log in'
    Write-Host '     - Click on the profile icon in the upper-right corner and select My Settings'
    Write-Host '     - In the Authentication tab, click `Download tripleblind.yaml`'
    Write-Host '     - Place the downloaded file into your SDK folder (replace the existing file)'
    Write-Host '    Alternatively, provide just your Authentication Token below.'

    $token = Read-Host -Prompt 'If you have set up your yaml file, press Enter. Or, provide your token'
}

if ($token) {
    $token = "sdkuser_token: `"$token`""
    (Get-Content tripleblind.yaml) -replace 'sdkuser_token: ""', $token | Set-Content tripleblind.yaml
    Ok; Write-Host 'token saved successfully'
}



#### WRAP UP ####
# output final information
if ($everythingOk) {
    Ok; Write-Host 'TripleBlind installation complete.'
    Write-Host '    Remember to activate the conda environment whenever working with it:'
    Write-Host '        conda activate tripleblind'
    Write-Host ' '
} else {
    Excl; Write-Host 'TripleBlind installation complete,'
    Write-Host '    but there may have been issues during installation.'
    Write-Host '    TripleBlind may not work properly'
    Write-Host '    Review the installer output carefully.'
    Write-Host '    Remember to activate the conda environment whenever working with it:'
    Write-Host '        conda activate tripleblind'
    Write-Host ' '
}

if (-not $ni) {
    $question = '    Do you want to open TripleBlind documentation?'
    $choices  = '&Yes, open the documentation', '&No, just close the installer'
    $decision = $Host.UI.PromptForChoice('', $question, $choices, 1)

    if ($decision -eq 0) {
        Invoke-Item README.html
    }
}

$ErrorActionPreference = $oldPreference
if ($DebugPreference) {
    Write-Host '    Debugging log can be found in the SDK folder as debug.txt'
    Stop-Transcript
}

exit

# SIG # Begin signature block
# MIIpngYJKoZIhvcNAQcCoIIpjzCCKYsCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDPBWwoUz3s6oEJ
# vl89FJ+3g9cukK86d24s5kIPg86nRKCCEo4wggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBvcwggVfoAMCAQICEAHRUuKYNE3WmSWEpo2+QVowDQYJ
# KoZIhvcNAQELBQAwVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIEVW
# IFIzNjAeFw0yMzExMDYwMDAwMDBaFw0yNDExMDUyMzU5NTlaMIG8MRMwEQYDVQQF
# EwpGMDAxMzMzMzEyMRMwEQYLKwYBBAGCNzwCAQMTAlVTMRkwFwYLKwYBBAGCNzwC
# AQITCE1pc3NvdXJpMR0wGwYDVQQPExRQcml2YXRlIE9yZ2FuaXphdGlvbjELMAkG
# A1UEBhMCVVMxETAPBgNVBAgMCE1pc3NvdXJpMRowGAYDVQQKDBFUcmlwbGVCbGlu
# ZCwgSW5jLjEaMBgGA1UEAwwRVHJpcGxlQmxpbmQsIEluYy4wggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQDQ9D9A1QOmtOEesta/CnrUZTyVpuwpxP9pEoYt
# AsnNcFzXV3ldyzMkM6hiUVWG9cER12fWtchpCxGcz0zW7ayC3CjuWsUFdNXaQDbt
# 29WK71w1Xcqldcafkq9fRQIUzO7wjdLSO+4QppcXo5rLED8IbDD1XUKPP1RAQovs
# oBavbv1MTY2lFhSCl/qjtfTg3xTifHNRAIg3IcdTjH87fnzR1lWd4+rJfgLaEEdg
# CizzHjW+0a7AQFWl0Rv7Ot0F0Y76CNq4sYjNsSjYTfv3A1hhBqiM0jRiSvEHopFf
# SXwmH7fJYUcluiIb7P986p2VsesrI+dc2tEwlWuNKyWezgP5pjiVtkjDwR+nmDDK
# JzvHgIC5u1aSFxSy9l4B45/osvPPR8038T7kfKEV4KcgW09zfNKMpn+4jMct+xNn
# v4vCIT5b8kGBEvqMJgzvk82PxMLm43WK5LxzfKkkTCwCZDuW3pYPcBhXJnjq413U
# bM7hTuS7V5Sq6skB34bjuBh8+232yXZvl2i5mXS2/s0mxdqqBPWiwGsNzg0W6b4S
# z0CVx3oWRs8hlTHw76k3BO0TR/O+yMrgzFs9b/CrumEWPqBu5fnFKQ+wj25nv/YG
# yOpYivVA+IzMmSC8BIRehEQNBUfaFp3cPJk+38YlURyrd9cr3qi4xJ4h+VWK2xBV
# FE084wIDAQABo4IB1zCCAdMwHwYDVR0jBBgwFoAUgTKSQSsozUbIxKLGKjkS7Eip
# PxQwHQYDVR0OBBYEFHwU3a/9R+JR/27ddyekE2p62QhAMA4GA1UdDwEB/wQEAwIH
# gDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUFBwMDMEkGA1UdIARCMEAw
# NQYMKwYBBAGyMQECAQYBMCUwIwYIKwYBBQUHAgEWF2h0dHBzOi8vc2VjdGlnby5j
# b20vQ1BTMAcGBWeBDAEDMEsGA1UdHwREMEIwQKA+oDyGOmh0dHA6Ly9jcmwuc2Vj
# dGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5nQ0FFVlIzNi5jcmwwewYI
# KwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8vY3J0LnNlY3RpZ28uY29t
# L1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ0NBRVZSMzYuY3J0MCMGCCsGAQUFBzAB
# hhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTBJBgNVHREEQjBAoCYGCCsGAQUFBwgD
# oBowGAwWVVMtTUlTU09VUkktRjAwMTMzMzMxMoEWZGV2b3BzQHRyaXBsZWJsaW5k
# LmNvbTANBgkqhkiG9w0BAQsFAAOCAYEAPyrfdvfBogSU5LmgyaYKGdpYy5kvugp9
# 6pMyvwmnkckO++b2m9K/7S+NeuYRS7RHcdZu+/bQu957LtXbKIO+3ZmJNe2w39t2
# hCjR5SPfrYBfXqtn+iWIKQRZFKMIRA5OHGzCYVaP/QSP78kLDyiuBH64df3TVeWA
# DaPLoaQBiGKu6X4FXSOHMQvJwCgnzxUzA6N6hxqhtkxMdovmidGHM76B3rIvfckP
# 4BPUa+fDxHdyMrbrdE2WqbvLA9YMfFiYJR5ym7AWJqO1BW2KHtJEDVea7TnN1MNZ
# aEa6FJlSPdMcohQEve86B50uGyFZPFwY1sC+sUhndilgCKEVuOblrqRaD1br/azo
# C/Hp4gNWpO/sju/FnDciT2O2WfbALWPEfWN5KZ9P38pZLD0u9I+2oG2qQsiY8yB4
# +sbKuGWWyjWkrUokPWuajpgfsXRps22b5srnW4V3EOtblDo0nZBipdc+wCkg34kJ
# B60IzsyM07gziYh6m3IW+ir7wlhBvDPWMYIWZjCCFmICAQEwazBXMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgQ0EgRVYgUjM2AhAB0VLimDRN1pklhKaNvkFa
# MA0GCWCGSAFlAwQCAQUAoHwwEAYKKwYBBAGCNwIBDDECMAAwGQYJKoZIhvcNAQkD
# MQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJ
# KoZIhvcNAQkEMSIEII9A8iIsJ1ItJJOVyg+UFnFwbO3Zknq0mPB9FXs8pkkoMA0G
# CSqGSIb3DQEBAQUABIICAARYD6hPdemmQ94e/w8s7ZRc0Zq4MjUm9/P9Z3gn3otE
# HAcPWhdvlBTD5e7Lf0VmcxynXyXIwKD+PnanMxi+LuNE7M3PoyH1U32xHBu+OO2Y
# 40LlyUHW/EwdzV4M+IbmW09+/RjOm06nw9Cv/RNSZV2yC96fJyWo+WcbXZmxBT8x
# SEwh/j+BAb4CDnMD+7eJzeM0Ilum2ExEtNTfKHdSlnu1mP9Hd1hc5Lht5MFxc3Sy
# fIqov7AECWWlz1vZ0vkWt+ROZFxF0Y0HMW05+sUSngkIzoPJS+zRqMNG9EvNR4Ne
# CnKCaiVjVmP7E0+NuqqEC52lQWgrFMABd0qySt5G73rcve7+6Cg8c4FmY3bpkDKi
# 3Tb1hKHX5d+g3TKT/nRcmwTzid6qQo6xkgdRwdmfBOD6LWyQ/DoUvP2yymp1B2Zn
# TrLnZk4gz3JtC3+20FzfeodabWk2g38F31q1jC8sEzKwX82Q5ARG5d0x3p+I+f1H
# tIrzy8o59arXXj++qWc4SJjislM3ipqybuVZ/c1vRbFUQ0FlqU6d0W9+9UiJCdgC
# ptUed+fnFHEQPty4BtINXVA5l0oqovHDs2+qLgbHymzfy/C50Kebe1YcYyalu9S3
# GilYepLooVIiWpSW+b/XsRC5f6syQh+ZFFyn5JSbWXcH9hLwz/StvGpmZVjuh3LM
# oYITTjCCE0oGCisGAQQBgjcDAwExghM6MIITNgYJKoZIhvcNAQcCoIITJzCCEyMC
# AQMxDzANBglghkgBZQMEAgIFADCB7wYLKoZIhvcNAQkQAQSggd8EgdwwgdkCAQEG
# CisGAQQBsjECAQEwMTANBglghkgBZQMEAgEFAAQgyGPXHrOgztFMOKRwp+h+9UIB
# gMgCwjTKcS57LG6evCkCFEyFK/7apNK/mcWPggRbfI4x/BupGA8yMDI0MDYyNzE5
# NDMyNFqgbqRsMGoxCzAJBgNVBAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgw
# FgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMMI1NlY3RpZ28gUlNBIFRp
# bWUgU3RhbXBpbmcgU2lnbmVyICM0oIIN6TCCBvUwggTdoAMCAQICEDlMJeF8oG0n
# qGXiO9kdItQwDQYJKoZIhvcNAQEMBQAwfTELMAkGA1UEBhMCR0IxGzAZBgNVBAgT
# EkdyZWF0ZXIgTWFuY2hlc3RlcjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMP
# U2VjdGlnbyBMaW1pdGVkMSUwIwYDVQQDExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1w
# aW5nIENBMB4XDTIzMDUwMzAwMDAwMFoXDTM0MDgwMjIzNTk1OVowajELMAkGA1UE
# BhMCR0IxEzARBgNVBAgTCk1hbmNoZXN0ZXIxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDEsMCoGA1UEAwwjU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBTaWduZXIg
# IzQwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQCkkyhSS88nh3akKRyZ
# OMDnDtTRHOxoywFk5IrNd7BxZYK8n/yLu7uVmPslEY5aiAlmERRYsroiW+b2MvFd
# LcB6og7g4FZk7aHlgSByIGRBbMfDCPrzfV3vIZrCftcsw7oRmB780yAIQrNfv3+I
# WDKrMLPYjHqWShkTXKz856vpHBYusLA4lUrPhVCrZwMlobs46Q9vqVqakSgTNbkf
# 8z3hJMhrsZnoDe+7TeU9jFQDkdD8Lc9VMzh6CRwH0SLgY4anvv3Sg3MSFJuaTAlG
# vTS84UtQe3LgW/0Zux88ahl7brstRCq+PEzMrIoEk8ZXhqBzNiuBl/obm36Ih9hS
# eYn+bnc317tQn/oYJU8T8l58qbEgWimro0KHd+D0TAJI3VilU6ajoO0ZlmUVKcXt
# MzAl5paDgZr2YGaQWAeAzUJ1rPu0kdDF3QFAaraoEO72jXq3nnWv06VLGKEMn1ew
# XiVHkXTNdRLRnG/kXg2b7HUm7v7T9ZIvUoXo2kRRKqLMAMqHZkOjGwDvorWWnWKt
# JwvyG0rJw5RCN4gghKiHrsO6I3J7+FTv+GsnsIX1p0OF2Cs5dNtadwLRpPr1zZw9
# zB+uUdB7bNgdLRFCU3F0wuU1qi1SEtklz/DT0JFDEtcyfZhs43dByP8fJFTvbq3G
# PlV78VyHOmTxYEsFT++5L+wJEwIDAQABo4IBgjCCAX4wHwYDVR0jBBgwFoAUGqH4
# YRkgD8NBd0UojtE1XwYSBFUwHQYDVR0OBBYEFAMPMciRKpO9Y/PRXU2kNA/SlQEY
# MA4GA1UdDwEB/wQEAwIGwDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsG
# AQUFBwMIMEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMIMCUwIwYIKwYBBQUHAgEW
# F2h0dHBzOi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEAjBEBgNVHR8EPTA7MDmg
# N6A1hjNodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29SU0FUaW1lU3RhbXBp
# bmdDQS5jcmwwdAYIKwYBBQUHAQEEaDBmMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0
# LnNlY3RpZ28uY29tL1NlY3RpZ29SU0FUaW1lU3RhbXBpbmdDQS5jcnQwIwYIKwYB
# BQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4IC
# AQBMm2VY+uB5z+8VwzJt3jOR63dY4uu9y0o8dd5+lG3DIscEld9laWETDPYMnvWJ
# IF7Bh8cDJMrHpfAm3/j4MWUN4OttUVemjIRSCEYcKsLe8tqKRfO+9/YuxH7t+O1o
# v3pWSOlh5Zo5d7y+upFkiHX/XYUWNCfSKcv/7S3a/76TDOxtog3Mw/FuvSGRGiMA
# Uq2X1GJ4KoR5qNc9rCGPcMMkeTqX8Q2jo1tT2KsAulj7NYBPXyhxbBlewoNykK7g
# xtjymfvqtJJlfAd8NUQdrVgYa2L73mzECqls0yFGcNwvjXVMI8JB0HqWO8NL3c2S
# JnR2XDegmiSeTl9O048P5RNPWURlS0Nkz0j4Z2e5Tb/MDbE6MNChPUitemXk7N/g
# AfCzKko5rMGk+al9NdAyQKCxGSoYIbLIfQVxGksnNqrgmByDdefHfkuEQ81D+5CX
# dioSrEDBcFuZCkD6gG2UYXvIbrnIZ2ckXFCNASDeB/cB1PguEc2dg+X4yiUcRD0n
# 5bCGRyoLG4R2fXtoT4239xO07aAt7nMP2RC6nZksfNd1H48QxJTmfiTllUqIjCfW
# hWYd+a5kdpHoSP7IVQrtKcMf3jimwBT7Mj34qYNiNsjDvgCHHKv6SkIciQPc9Vx8
# cNldeE7un14g5glqfCsIo0j1FfwET9/NIRx65fWOGtS5QDCCBuwwggTUoAMCAQIC
# EDAPb6zdZph0fKlGNqd4LbkwDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwG
# A1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3Qg
# UlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE5MDUwMjAwMDAwMFoXDTM4
# MDExODIzNTk1OVowfTELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFu
# Y2hlc3RlcjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1p
# dGVkMSUwIwYDVQQDExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIENBMIICIjAN
# BgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAyBsBr9ksfoiZfQGYPyCQvZyAIVST
# uc+gPlPvs1rAdtYaBKXOR4O168TMSTTL80VlufmnZBYmCfvVMlJ5LsljwhObtoY/
# AQWSZm8hq9VxEHmH9EYqzcRaydvXXUlNclYP3MnjU5g6Kh78zlhJ07/zObu5pCNC
# rNAVw3+eolzXOPEWsnDTo8Tfs8VyrC4Kd/wNlFK3/B+VcyQ9ASi8Dw1Ps5EBjm6d
# J3VV0Rc7NCF7lwGUr3+Az9ERCleEyX9W4L1GnIK+lJ2/tCCwYH64TfUNP9vQ6oWM
# ilZx0S2UTMiMPNMUopy9Jv/TUyDHYGmbWApU9AXn/TGs+ciFF8e4KRmkKS9G493b
# kV+fPzY+DjBnK0a3Na+WvtpMYMyou58NFNQYxDCYdIIhz2JWtSFzEh79qsoIWId3
# pBXrGVX/0DlULSbuRRo6b83XhPDX8CjFT2SDAtT74t7xvAIo9G3aJ4oG0paH3uhr
# DvBbfel2aZMgHEqXLHcZK5OVmJyXnuuOwXhWxkQl3wYSmgYtnwNe/YOiU2fKsfqN
# oWTJiJJZy6hGwMnypv99V9sSdvqKQSTUG/xypRSi1K1DHKRJi0E5FAMeKfobpSKu
# pcNNgtCN2mu32/cYQFdz8HGj+0p9RTbB942C+rnJDVOAffq2OVgy728YUInXT50z
# vRq1naHelUF6p4MCAwEAAaOCAVowggFWMB8GA1UdIwQYMBaAFFN5v1qqK0rPVIDh
# 2JvAnfKyA2bLMB0GA1UdDgQWBBQaofhhGSAPw0F3RSiO0TVfBhIEVTAOBgNVHQ8B
# Af8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcD
# CDARBgNVHSAECjAIMAYGBFUdIAAwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2Ny
# bC51c2VydHJ1c3QuY29tL1VTRVJUcnVzdFJTQUNlcnRpZmljYXRpb25BdXRob3Jp
# dHkuY3JsMHYGCCsGAQUFBwEBBGowaDA/BggrBgEFBQcwAoYzaHR0cDovL2NydC51
# c2VydHJ1c3QuY29tL1VTRVJUcnVzdFJTQUFkZFRydXN0Q0EuY3J0MCUGCCsGAQUF
# BzABhhlodHRwOi8vb2NzcC51c2VydHJ1c3QuY29tMA0GCSqGSIb3DQEBDAUAA4IC
# AQBtVIGlM10W4bVTgZF13wN6MgstJYQRsrDbKn0qBfW8Oyf0WqC5SVmQKWxhy7VQ
# 2+J9+Z8A70DDrdPi5Fb5WEHP8ULlEH3/sHQfj8ZcCfkzXuqgHCZYXPO0EQ/V1cPi
# vNVYeL9IduFEZ22PsEMQD43k+ThivxMBxYWjTMXMslMwlaTW9JZWCLjNXH8Blr5y
# Umo7Qjd8Fng5k5OUm7Hcsm1BbWfNyW+QPX9FcsEbI9bCVYRm5LPFZgb289ZLXq2j
# K0KKIZL+qG9aJXBigXNjXqC72NzXStM9r4MGOBIdJIct5PwC1j53BLwENrXnd8uc
# Lo0jGLmjwkcd8F3WoXNXBWiap8k3ZR2+6rzYQoNDBaWLpgn/0aGUpk6qPQn1BWy3
# 0mRa2Coiwkud8TleTN5IPZs0lpoJX47997FSkc4/ifYcobWpdR9xv1tDXWU9UIFu
# q/DQ0/yysx+2mZYm9Dx5i1xkzM3uJ5rloMAMcofBbk1a0x7q8ETmMm8c6xdOlMN4
# ZSA7D0GqH+mhQZ3+sbigZSo04N6o+TzmwTC7wKBjLPxcFgCo0MR/6hGdHgbGpm0y
# XbQ4CStJB6r97DDa8acvz7f9+tCjhNknnvsBZne5VhDhIG7GrrH5trrINV0zdo7x
# fCAMKneutaIChrop7rRaALGMq+P5CslUXdS5anSevUiumDGCBCwwggQoAgEBMIGR
# MH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAO
# BgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDElMCMGA1UE
# AxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQQIQOUwl4XygbSeoZeI72R0i
# 1DANBglghkgBZQMEAgIFAKCCAWswGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEE
# MBwGCSqGSIb3DQEJBTEPFw0yNDA2MjcxOTQzMjRaMD8GCSqGSIb3DQEJBDEyBDAF
# rJazZLXMbmpXm1z1lnedli9MYY0i5qr4oObS0Hlu61cADjI7Qor3XWKjkqWjI7ww
# ge0GCyqGSIb3DQEJEAIMMYHdMIHaMIHXMBYEFK5ir3UKDL1H1kYfdWjivIznyk+U
# MIG8BBQC1luV4oNwwVcAlfqI+SPdk3+tjzCBozCBjqSBizCBiDELMAkGA1UEBhMC
# VVMxEzARBgNVBAgTCk5ldyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBDaXR5MR4w
# HAYDVQQKExVUaGUgVVNFUlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVTRVJUcnVz
# dCBSU0EgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkCEDAPb6zdZph0fKlGNqd4Lbkw
# DQYJKoZIhvcNAQEBBQAEggIABBx8KQuEsKwFAu/PiEUcE7MutbC44LJgs/dC7LSU
# SkCmSTQQRFVUiO8NgD530Pygj3konNQgBn8deSz2zD7KYTrgjrEbQijWpwJCp5nO
# 0SCcRJGGk6783m5wKmqukVVXo5Lt2CHVCczBmqvTFKB1HGbIsbnWHKG5vmh3azR4
# SbntcnxgpdJOk39cdo5JczzmV5gg73HT6pnvSgzNM4dVTE4MF/6SVy+tYh408IvE
# 2En1NV7q+XR3naBfmSZ6en0qVXKt8Cis+FQjTbXVDoeSm309uegvy8lL/aK1rObw
# i547E0qUQalCrzeKU7Ya1L3xP783UUtJWiogwtaFiOIfA8kAn78eIQnOVZ3IBhIz
# EebI2B9RFyjHCWF8IP7l2oParkpqDxWrvUGXinVdY0dQVrlBVngHkSe/u1l1pm3K
# qGVj9M+CR4s3KzxgpIhPyViOzw9QaXlGZnqtgfFdNE3Ej+HSE1V6vUulW8AmDLkf
# e/cXBAELPCf3B/01baYcZh1ZAj0DhYtPZ5zt1xJrNmhKT2Kj8j5hurIzdXC6S7m9
# UMSmfGx5LwS6YTfpiBTqpMSuYMj3w0DzFU46HqFlYMieuqqIxHCFBOhH0GnI7jhk
# +315YBDgT+Ks6/kqYrSVaagNEk1ZsDNch81cdzHPtUTxC9WQddCiUEdtEXujwe8O
# ZnU=
# SIG # End signature block
