#!/usr/bin/env Rscript
# Copyright (c) TripleBlind Holdings, Inc. Confidential and Proprietary. All rights reserved.

# Install required packages if not already installed
RequiredPackages <- c("reticulate", "stringr")
for (i in RequiredPackages) { # Installs packages if not yet installed
  if (!require(i, character.only = TRUE)) install.packages(i)
}

# Load packages
library(reticulate)
library(stringr)

#Set relevant file paths
curr_file_path <- rstudioapi::getSourceEditorContext()$path
cwd <- dirname(curr_file_path)
setwd(cwd)

data_dir <- file.path("example_data")

tb <- import("tripleblind")

# The data pre-process stage has not yet been converted to R.
# This step downloads files necessary to run the example
tb$util$download_tripleblind_resource(
    "SAN.csv",
    save_to_dir=data_dir,
    cache_dir="../../../.cache"
)
tb$util$download_tripleblind_resource(
    "JPM.csv",
    save_to_dir=data_dir,
    cache_dir="../../../.cache"
)
tb$util$download_tripleblind_resource(
    "PNB.csv",
    save_to_dir=data_dir,
    cache_dir="../../../.cache"
)


# Unique value used by all scripts in this folder.  Edit "run_id.txt" to change
run_id <- tb$util$read_run_id()

# IN THIS INSTANCE WE ARE TREATING ORGANIZATION-ONE AS "SANTANDER"
#
# Establish the connection details to reach the TripleBlind instance.
# Unless explicitly specified, all operations will occur via this default
# session as the user 'organization_one'
tb$initialize(api_token = tb$config$example_user1[["token"]])

##############################################################################
# FOR BREVITY, WE WILL CREATE DATA FOR ALL THREE COMPANIES IS ONE JOB.
# IN A REAL SITUATION THESE WOULD BE INDEPENDENT JOBS RUN BY THE SEPARATE
# ORGANIZATIONS


##############################################################################
#  Company one - simulating data from Santander
#############################################################################

# Create the first training dataset to the Santander Access Point

result <- tryCatch(
  {
    # Start a connection as 'organization_one' user
    session1 <- tb$Session(api_token = tb$config$example_user1[["token"]], from_default = TRUE)

    # Make data available on the Router.  Position data on the Access Point and
    # save meta-data about the dataset on the Router.
    #
    print(
      paste("Creating dataset 'SAN", run_id, "' on ", tb$config$example_user1[["login"]], "'s Access Point...", sep = "")
    )
    dataset_train0 <- tb$Asset$position(
      file_handle = file.path(data_dir, "SAN.csv"),
      name = paste("SAN", run_id, sep = ""),
      desc = paste("SAN", run_id, " dataset used in the Tabular_Data and XGBoost examples.", sep = ""),
      is_discoverable = TRUE,
      session = session1
    )
  },
  error = function(err) {
    err_type <- str_extract(err[[1]], "TripleblindAssetAlreadyExists") # err is a list

    if (is.na(err_type[[1]])) {
      print(paste("An error occurred: ", err, ""))
      cat("\n")
      quit(save = "ask")
    } else {
      if (err_type == "TripleblindAssetAlreadyExists") {
        print(paste("   asset 'SAN", run_id, "' already exists, skipping.", sep = ""))
      }
    }
  }
)

#############################################################################
#  Company two - simulating data from JPMorgan
#############################################################################

# Creating organization-two dataset to the JPMorgan Access Point

result <- tryCatch(
  {
    # Start a connection as 'organization_two' user
    session2 <- tb$Session(api_token = tb$config$example_user2[["token"]], from_default = TRUE)

    # Make data available on the Router.  Position data on the Access Point and
    # save meta-data about the dataset on the Router.
    #
    print(
      paste("Creating dataset 'JPM", run_id, "' on ", tb$config$example_user2[["login"]], "'s Access Point...", sep = "")
    )
    dataset_train1 <- tb$Asset$position(
      file_handle = file.path(data_dir, "JPM.csv"),
      name = paste("JPM", run_id, sep = ""),
      desc = paste("JPM", run_id, " dataset used in the Tabular_Data and XGBoost examples.", sep = ""),
      is_discoverable = TRUE,
      session = session2
    )

    if (tb$config$create_agreements_on_example_input_assets) {
      # For this example we will attach an Agreement to the datasets.  This
      # agreement makes the dataset available to the other team,
      # meaning the training step will not require the dataset owner to
      # explicitly grant permission for use.
      dataset_train1$add_agreement(
        with_team = tb$config$example_user1[["team_id"]],
        operation = tb$Operation$BLIND_LEARNING,
        session = session2
      )
      print("Created Agreement")
    }
  },
  error = function(err) {
    err_type <- str_extract(err[[1]], "TripleblindAssetAlreadyExists") # err is a list

    if (is.na(err_type[[1]])) {
      print(paste("An error occurred: ", err, ""))
      cat("\n")
      quit(save = "ask")
    } else {
      if (err_type == "TripleblindAssetAlreadyExists") {
        print(paste("   asset 'JPM", run_id, "' already exists, skipping.", sep = ""))
      }
    }
  }
) # end trycatch

#############################################################################
#  Company three - simulating data from PNB Paribas
#############################################################################


result <- tryCatch(
  {
    # Start a connection as 'organization_three' user
    session3 <- tb$Session(api_token = tb$config$example_user3[["token"]], from_default = TRUE)

    # Make data available on the Router.  Position data on the Access Point and
    # save meta-data about the dataset on the Router.
    #
    print(
      paste("Creating dataset 'PNB'", run_id, "' on ", tb$config$example_user3[["login"]], "'s Access Point...", sep = "")
    )
    dataset_train2 <- tb$Asset$position(
      file_handle = file.path(data_dir, "PNB.csv"),
      name = paste("PNB", run_id, sep = ""),
      desc = paste("PNB", run_id, " dataset used in the Tabular_Data and XGBoost examples.", sep = ""),
      is_discoverable = TRUE,
      session = session3
    )

    if (tb$config$create_agreements_on_example_input_assets) {
      # Create Agreement
      dataset_train2$add_agreement(
        with_team = tb$config$example_user1[["team_id"]],
        operation = tb$Operation$BLIND_LEARNING,
        session = session3
      )
      print("Created Agreement")
    }
  },
  error = function(err) {
    err_type <- str_extract(err[[1]], "TripleblindAssetAlreadyExists") # err is a list

    if (is.na(err_type[[1]])) {
      print(paste("An error occurred: ", err, ""))
      cat("\n")
      quit(save = "ask")
    } else {
      if (err_type == "TripleblindAssetAlreadyExists") {
        print(paste("   asset 'PNB", run_id, "' already exists, skipping.", sep = ""))
      }
    }
  }
) # end trycatch

print("Data is in position.")
