#!/usr/bin/env Rscript
# Copyright (c) TripleBlind Holdings, Inc. Confidential and Proprietary. All rights reserved.

# Install required packages if not already installed
RequiredPackages <- c("reticulate", "stringr")
for (i in RequiredPackages) { # Installs packages if not yet installed
  if (!require(i, character.only = TRUE)) install.packages(i)
}

# Load packages
library(reticulate)
library(stringr)

#Set relevant file paths
curr_file_path <- rstudioapi::getSourceEditorContext()$path
cwd <- dirname(curr_file_path)
setwd(cwd)

py_file_path<-file.path("..", "..", "Blind_Join", "0_build_fake_data.py")
data_dir <- file.path("example_data")

# The data pre-process stage has not yet been converted to R.
# This step runs the 0_build_fake_data file in python.
if (!file.exists(file.path(data_dir, "store_transactions.csv")) ||
  !file.exists(file.path(data_dir, "passengers.csv"))) {
  py_run_file(py_file_path)
}

tb <- import("tripleblind")

run_id <- tb$util$read_run_id()

tb$util$set_script_dir_current()
tb$initialize(api_token = tb$config$example_user1[["token"]])

result <- tryCatch(
  {
    print(
      paste("Uploading 'Shop Transaction-", run_id, "' to ", tb$config$example_user1[["login"]], "'s Access Point...", sep = "")
    )

    asset_0 <- tb$Asset$position(
      file_handle = file.path(data_dir, "store_transactions.csv"),
      name = paste("Shop Transaction-", run_id, sep = ""),
      desc = "Fictional retail transaction data.",
      is_discoverable = TRUE,
    )
  },
  error = function(err) {
    err_type <- str_extract(err[[1]], "TripleblindAssetAlreadyExists") # err is a list

    if (is.na(err_type[[1]])) {
      print(paste("An error occurred: ", err, "")) # prints error if doesn't match type TripleblindAssetAlreadyExists
    } else {
      if (err_type == "TripleblindAssetAlreadyExists") {
        print(
          paste("Asset 'Shop Transaction-", run_id, "' already exists, change value in run_id.txt to make it unique.'", sep = "")
        )
      }
    }
  }
) # End trycatch

session_org_2 <- tb$Session(
  api_token = tb$config$example_user2[["token"]], from_default = TRUE
)

use_simple_dataset <- TRUE

if (use_simple_dataset) {
  # The default way of running this example uses a simple table asset
  result <- tryCatch(
    {
      print(
        paste("Uploading 'Transport Data-", run_id, "' to ", tb$config$example_user2[["login"]], "'s Access Point...", sep = "")
      )
      asset_1 <- tb$Asset$position(
        file_handle = file.path(data_dir, "passengers.csv"),
        name = paste("Transport Data-", run_id, sep = ""),
        desc = "Fictional passenger transport data",
        is_discoverable = TRUE,
        session = session_org_2
      )

      if (tb$config$create_agreements_on_example_input_assets) {
        # For this example we will attach an Agreement to the datasets. The
        # agreement makes the dataset available to the other team,
        # meaning the training step will not require the dataset owner to
        # explicitly grant permission for use.
        asset_1$add_agreement(
          with_team = tb$config$example_user1[["team_id"]],
          operation = tb$Operation$BLIND_JOIN,
          session = session_org_2
        )
        print("Created Agreement")
      }
    },
    error = function(err) {
      err_type <- str_extract(err[[1]], "TripleblindAssetAlreadyExists") # err is a list

      if (is.na(err_type[[1]])) {
        print(paste("An error occurred: ", err, "")) # prints error if doesn't match type TripleblindAssetAlreadyExists
      } else {
        if (err_type == "TripleblindAssetAlreadyExists") {
          print(
            paste("Asset 'Transport Data-", run_id, "' already exists, change value in run_id.txt to make it unique.'", sep = "")
          )
        }
      }
    }
  ) # end trycatch
} else {
  # An alternate method would be to expose a limited subset of data from a
  # live database.  An SQL query would generate a view on the minimal fields
  # necessary to perform the operations, reducing the chances of any leakage
  # of the passenger database information from human error (e.g. approving
  # consumer access to more fields than intended).

  # This example illustrates using a BigQuery dataset as the live database

  query <- "SELECT customer_id, depart_station, arrive_station from `your-projectid.your_datasets.passengers`"

  # NOTE: You will need to create you own database under your own account to
  #       run the example this way.  You can populate a 'passengers' table
  #       with the data from "passengers.csv".
  credentials <- Sys.getenv("GOOGLE_APPLICATION_CREDENTIALS")
  gcp_project_id <- "your-projectid"
  bigquery_dataset <- "your_datasets"

  # Create an asset on the Router that exposes a limed subset of fields in a
  # database hosted on BigQuery.

  result <- tryCatch(
    {
      asset_1 <- tb$asset$BigQueryDatabase$create(
        gcp_project_id,
        bigquery_dataset,
        credentials,
        query = query,
        name = paste("Transport Data-", run_id, sep = ""),
        desc = "
                        Limited view of the transportation company's database.  This
                        view only includes the fields the transportation company is
                        willing to share with outside organizations for statistical
                        analysis.",
        is_discoverable = TRUE,
        session = session_org_2
      )

      if (tb$config$create_agreements_on_example_input_assets) {
        asset_1$add_agreement(
          with_team = tb$config$example_user1[["team_id"]],
          operation = tb$Operation$BLIND_JOIN,
          session = session_org_2
        )
        print("Created Agreement")
      }
    },
    error = function(err) {
      err_type <- str_extract(err[[1]], "TripleblindAssetAlreadyExists") # err is a list

      if (is.na(err_type[[1]])) {
        print(paste("An error occurred: ", err, "")) # prints error if doesn't match type TripleblindAssetAlreadyExists
      } else {
        if (err_type == "TripleblindAssetAlreadyExists") {
          print(
            paste("Asset 'Transport Data-", run_id, "' already exists, change value in run_id.txt to make it unique.'", sep = "")
          )
        }
      }
    }
  ) # end trycatch
}

print("Data is in position.")
